"use client"

import { Button } from "@/components/ui/button"
import { Table, TableBody, TableCell, TableRow } from "@/components/ui/table"
import { cn } from "@/lib/utils"
import { ComponentWithClassName, ComponentWithClassNameAndChildren } from "@/types/utils"
import { ChevronLeftIcon, ChevronRightIcon, SymbolIcon } from "@radix-ui/react-icons"
import { flexRender, getCoreRowModel, getPaginationRowModel, getSortedRowModel, useReactTable, type SortingState, type Table as TanstackTable } from "@tanstack/react-table"
import { useState } from "react"
import { TransformedUsers } from "../type/transformers"
import { columns } from "./users.table-columns"
import { WorkflowButtonInvite } from "./workflow.button-invite"
import { FadeIn } from "@/components/animations/fade-in"
import { INTERNAL__BASE_FADE_IN_DELAY } from "../lib/constants"

export function UsersTable({ data, className }: ComponentWithClassName<{ data: NonNullable<TransformedUsers> }>) {
  const [sorting, setSorting] = useState<SortingState>([])
  const [pagination, setPagination] = useState({
    pageIndex: 0,
    pageSize: 10,
  });
  const table = useReactTable({
    data,
    columns,
    getCoreRowModel: getCoreRowModel(),
    getPaginationRowModel: getPaginationRowModel(),
    getSortedRowModel: getSortedRowModel(),
    onSortingChange: setSorting,
    onPaginationChange: setPagination,
    state: {
      sorting,
      pagination,
    }
  })

  return (
    <div className={cn("flex flex-col gap-0", className)}>
      <div className="grid gap-8">
        <SizableContentWrapper>
          <INTERNAL__DataTable
            table={table}
          />
        </SizableContentWrapper>
      </div>

      {table.getPageCount() > 1 && (
        <div className="flex items-center justify-end space-x-2 py-4 p-6 pt-0">
          <div className="flex-1 text-sm text-muted-foreground">
            Showing{" "}
            {pagination.pageIndex + 1} of{" "}
            {table.getPageCount()} page(s).
          </div>

          <Button
            variant="ghost"
            size="icon"
            onClick={() => table.resetSorting()}
            className={sorting.length > 0 ? "text-muted-foreground" : "hidden"}
          >
            <div className="flex flex-row -space-x-2">
              <SymbolIcon className="size-4" />
            </div>
          </Button>
          <Button
            variant="ghost"
            size="icon"
            onClick={() => table.firstPage()}
            disabled={!table.getCanPreviousPage()}
          >
            <div className="flex flex-row -space-x-2">
              <ChevronLeftIcon className="size-4" />
              <ChevronLeftIcon className="size-4" />
            </div>
          </Button>
          <Button
            variant="ghost"
            size="icon"
            onClick={() => table.previousPage()}
            disabled={!table.getCanPreviousPage()}
          >
            <div className="flex flex-row -space-x-2">
              <ChevronLeftIcon className="size-4" />
            </div>
          </Button>
          <Button
            variant="ghost"
            size="icon"
            onClick={() => table.nextPage()}
            disabled={!table.getCanNextPage()}
          >
            <div className="flex flex-row -space-x-2">
              <ChevronRightIcon className="size-4" />
            </div>
          </Button>
          <Button
            variant="ghost"
            size="icon"
            onClick={() => table.lastPage()}
            disabled={!table.getCanNextPage()}
          >
            <div className="flex flex-row -space-x-2">
              <ChevronRightIcon className="size-4" />
              <ChevronRightIcon className="size-4" />
            </div>
          </Button>
        </div>
      )}
    </div>
  )
}

interface INTERNAL__DataTableProps<TData> {
  table: TanstackTable<TData>
}

function INTERNAL__DataTable<TData>({
  table
}: INTERNAL__DataTableProps<TData>) {

  return (
    <div>
      <div className="rounded-md border">
        <Table>
          <TableBody>
            {table.getRowModel().rows?.length ? (
              <>
                {
                  table.getRowModel().rows.map((row, i) => (
                    <TableRow
                      key={row.id}
                      className="[&>td:first-child]:pl-6"
                      data-state={row.getIsSelected() && "selected"}
                    >
                      {row.getVisibleCells().map((cell, ii) => (
                        <TableCell key={cell.id} className="last-of-type:max-w-none bg-background">
                          <FadeIn delay={INTERNAL__BASE_FADE_IN_DELAY + (0.05 * (i + ii))}>
                            {flexRender(cell.column.columnDef.cell, cell.getContext())}
                          </FadeIn>
                        </TableCell>
                      ))}
                    </TableRow>
                  ))
                }

                <TableRow className="m-0 transition-colors hover:bg-muted/50 data-[state=selected]:bg-muted rounded-none">
                  <TableCell colSpan={columns.length} className="p-0 pr-0 rounded-none">
                    <WorkflowButtonInvite
                      className={"w-full"}
                    />
                  </TableCell>
                </TableRow>
              </>
            ) : (
              <>
                <TableRow>
                  <TableCell colSpan={columns.length} className="h-24 text-center">
                    <div
                      className={"w-full h-full border border-dashed rounded place-content-center border-border"}
                    >
                      <p className="text-xs text-muted-foreground text-center">
                        No results.
                      </p>
                    </div>
                  </TableCell>
                </TableRow>

                <TableRow className="m-0 transition-colors hover:bg-muted/50 data-[state=selected]:bg-muted rounded-none">
                  <TableCell colSpan={columns.length} className="p-0 pr-0 rounded-none">
                    <WorkflowButtonInvite
                      className={"w-full"}
                    />
                  </TableCell>
                </TableRow>
              </>
            )}
          </TableBody>
        </Table>
      </div>
    </div>
  )
}

function SizableContentWrapper({ className, children }: ComponentWithClassNameAndChildren) {
  return (
    <div className={cn("overflow-x-auto", className)}>
      {children}
    </div>
  )
}

